/*
  indent fma_dfs_route.c -bli0 -bfda -kr -psl
*/

#include <time.h>

#include "libfma.h"
#include "lf_scheduler.h"
#include "lf_fabric.h"
#include "lf_fms_comm.h"
#include "lf_topo_map.h"
#include "libmyri.h"

#include "fma.h"
#include "fma_myri.h"
#include "fma_map.h"
#include "fma_fabric.h"

#include "fma_dfs_route.h"

/*
 * Local prototypes
 */
static void fma_dfs_clear_xbar_seen(struct lf_fabric *fp);
static void fma_dfs_clear_nic_route_count(struct lf_fabric *fp);
static int fma_dfs_get_direction(struct lf_xbar *from, struct lf_xbar *to);
static void fma_dfs_start_port_route(struct fma_nic_info *nip,
  int port, int current_route, int max_route_len);


#define FMA_DF_A_FEW_TIMES 10
#define MAX_XBAR_PORTS 64	/*buy some time */
enum { UP, DOWN, FLAT };

static void
fma_dfs_clear_xbar_seen(
  struct lf_fabric *fp)
{
  int i;

  for (i = 0; i < fp->num_xbars; i++) {
    FMA_XBAR(fp->xbars[i])->dfs_seen = 0;
    FMA_XBAR(fp->xbars[i])->dfs_up_skip_depth = 99;
    FMA_XBAR(fp->xbars[i])->dfs_down_skip_depth = 99;
  }
}

static void
fma_dfs_clear_nic_route_count(
  struct lf_fabric *fp)
{
  int i;

  for (i = 0; i < FMA_FABRIC(fp)->num_nics; i++) {
    memset(FMA_NIC(FMA_FABRIC(fp)->nics[i])->dfs_route_count, 0, 
	LF_MAX_NIC_PORTS * A.myri->nic_ports);
  }
}

/*get clos direction from 'from' to 'to'*/

static int
fma_dfs_get_direction(
  struct lf_xbar *from,
  struct lf_xbar *to)
{
  if (to->clos_level < from->clos_level)
    return DOWN;
  else if (to->clos_level > from->clos_level)
    return UP;
  else
    return FLAT;
}

static void
fma_dfs_clear_load(struct lf_fabric *fp)
{
  int i, j;

  for (i = 0; i < fp->num_xbars; i++) {
    for (j = 0; j < fp->xbars[i]->num_ports; j++) {
      FMA_XBAR(fp->xbars[i])->dfs_link_load[j] = 0;
    }

    fma_shuffle_port_order (fp->xbars[i], FMA_XBAR(fp->xbars[i])->dfs_port_order);
  }
}

static inline void
fma_dfs_increment_link_load(
  struct lf_xbar *xp,
  int port_index)
{
  struct fma_xbar *fxp;
  int pi2;
  int port;

  fxp = FMA_XBAR(xp);
  port = fxp->dfs_port_order[port_index];

  ++fxp->dfs_link_load[port];

  pi2 = port_index+1;
  while (pi2 < xp->num_ports &&
      fxp->dfs_link_load[(int) fxp->dfs_port_order[pi2]]
       < fxp->dfs_link_load[port]) {
    ++pi2;
  }

  --pi2;
  if (pi2 != port_index) {
    int t;

    t = fxp->dfs_port_order[pi2];
    fxp->dfs_port_order[pi2] = fxp->dfs_port_order[port_index];
    fxp->dfs_port_order[port_index] = t;
  }
}

int
fma_route_unique(
  unsigned char *route,
  int route_len,
  struct lf_nic *rmt_nic,
  int ri0,
  int rindex)
{
  int i;
  unsigned char *rp;
  unsigned char *route_buf;

  route_buf = FMA_NIC(rmt_nic)->route_buf;

  for (i=ri0; i<rindex; ++i) {
    rp = route_buf + FMA_ROUTE_OFFSET(i);
    if (memcmp(route, rp, route_len) == 0) {
      return 0;
    }
  }
  return 1;
}

/*
  does a dfs from 'x' stopping after 'sought' hosts are found.
  sets routes at each of these hosts.
  returns number of hosts found
  the dfs is limited by up/down. 
  port order is sorted by least use.
  only hosts found beyond depth are ignored.
*/
static int
fma_dfs(
  struct fma_nic_info *nip,
  int port,
  struct lf_xbar *xp,
  int depth,
  int direction,
  int route_num,
  unsigned char *route,
  int in_port,
  int max_route_len,
  int *unique_skipped)
{
  int i;
  int found;
  char ports[MAX_XBAR_PORTS];
  struct lf_xbar *rxp;
  struct fma_xbar *frxp;
  int rmt_np_index;
  int np_index;

  found = 0;
  *unique_skipped = 0;

  FMA_XBAR(xp)->dfs_seen = 1;

  np_index = FMA_NICPORT_INDEX(nip->nic_index, port);
  memcpy(ports, FMA_XBAR(xp)->dfs_port_order, xp->num_ports);

  for (i = 0; i < xp->num_ports && found == 0; i++) {
    union lf_node *rmt_node;   /* node that port i of 'xp' points to */
    int rmt_port;              /* port on 'rmt_node' */
    int pi;

    pi = ports[i];

    /* don't violate quadrant disable */
    if (lf_xbar_qd_violation(xp, in_port, pi)) continue;

    /* follow link to next node */
    rmt_node = xp->topo_ports[pi];
    rmt_port = xp->topo_rports[pi];
    rmt_np_index = (np_index * LF_MAX_NIC_PORTS) + rmt_port;

    /* If we cannot go this way, skip it */
    if (rmt_node == NULL || xp->link_state[pi] != LF_LINK_STATE_UP) {
      continue;
    }

    if (rmt_node->ln_type == LF_NODE_NIC) {
      struct lf_nic *rmt_nic;

      rmt_nic = LF_NIC(rmt_node);

      if (FMA_NIC(rmt_nic)->dfs_route_count[rmt_np_index] <= route_num) {
	int ri0;
	int rindex;
	unsigned char *nic_route;

	ri0 = FMA_ROUTE_INDEX(nip->nic_index, port, rmt_port);
	rindex = ri0 + route_num;

	if (!fma_route_unique(route, depth, rmt_nic, ri0, rindex)) {
	    ++*unique_skipped;
	    continue;
	}

	/*found a new host. save route */
	nic_route = FMA_NIC_N(rmt_node)->route_buf + FMA_ROUTE_OFFSET(rindex);

	memcpy(nic_route, route, depth);
	nic_route[depth] = LF_DELTA_TO_ROUTE(pi - in_port);
	FMA_NIC_N(rmt_node)->route_lens[rindex] = depth + 1;
	++FMA_NIC_N(rmt_node)->dfs_route_count[rmt_np_index];
	found++;
      }
    } else if (rmt_node->ln_type == LF_NODE_XBAR &&
	       !FMA_XBAR_N(rmt_node)->dfs_seen) {
      int count;
      int new_dir;
      int his_unique_skip;

      /* If this makes too long a route, do not descend */
      if (depth+1 >= max_route_len) continue;

      rxp = LF_XBAR(rmt_node);
      frxp = FMA_XBAR(rxp);

      /*obey the law */
      new_dir = fma_dfs_get_direction(xp, rxp);
      if (direction != UP && new_dir != DOWN) {
	continue;
      }

      if (new_dir == DOWN) {
	if (frxp->dfs_down_skip_depth <= depth) {
	  continue;
	}
      } else if (new_dir == UP) {
	if (frxp->dfs_up_skip_depth <= depth) {
	  continue;
	}
      }

      /* record route to this xbar */
      route[depth] = LF_DELTA_TO_ROUTE(pi - in_port);

      /* search down him */
      count = fma_dfs(nip, port, rxp, depth + 1,
        new_dir, route_num, route, rmt_port, max_route_len,
	&his_unique_skip);

      /* If nothing found, prune this xbar in this situation in the future */
      if (count == 0 && his_unique_skip == 0) {
	if (new_dir == UP) {
	  frxp->dfs_up_skip_depth = depth;
	} else if (new_dir == DOWN) {
	  frxp->dfs_down_skip_depth = depth;
	}
      } else {
	/* record load */
	fma_dfs_increment_link_load(xp, i);
	found += count;
      }
    }
  }

  /* Clear seen for this xbar */
  FMA_XBAR(xp)->dfs_seen = 0;
  return found;
}

static void
fma_dfs_start_port_route(
  struct fma_nic_info *nip,
  int port,
  int current_route,
  int max_route_len)
{
  struct lf_fabric *fp;
  union lf_node *np;
  struct lf_nic *nicp;
  struct lf_xbar *xp;
  unsigned char route_buf[MYRI_MAX_ROUTE_LEN];
  int found;
  int his_unique_skip;

  fp = A.fabric;

  /* Done already if not in map */
  if (nip->ni_map_index == -1) {
    return;
  }

  nicp = FMA_FABRIC(fp)->nics[nip->ni_map_index];
  np = nicp->topo_ports[port];

  /* If disconnected, nothing to do! */
  if (nicp->link_state[port] != LF_LINK_STATE_UP || np == NULL) {
    return;
  }

  /* If directly connected, record 0-length route */
  if (np->ln_type == LF_NODE_NIC) {
    struct lf_nic *rnicp;
    int rport;
    int rindex;

    rnicp = LF_NIC(np);
    rport = nicp->topo_rports[port];
   
    rindex = FMA_ROUTE_INDEX(nip->nic_index, port, rport) + current_route;
    FMA_NIC(rnicp)->route_lens[rindex] = 0;
    ++FMA_NIC(rnicp)->dfs_route_count[rport];

  } else {
    xp = LF_XBAR(np);

    fma_dfs_clear_xbar_seen(fp);

    do {
      found = fma_dfs(nip, port, xp, 0, UP, current_route,
	  route_buf, nip->nic_ptr->topo_rports[port], max_route_len,
	  &his_unique_skip);

    } while (found != 0);
  }
}

/*
  calculate a set of routes for all nics on this host
  clear_loads should be 1 the 1st time this is called, and 0 for subsequent
  calls if computing multiple routes.
*/
void
fma_dfs_calc_all_routes(
  int num_routes)
{
  struct fma_nic_info *nip;
  struct lf_fabric *fp;
  time_t route_start_time;
  int max_route_len[] = {1, 3, 7};
  int route_num;
  int port;
  int i;
  int n;
  
  fp = A.fabric;

  if (A.debug) {
    fma_log("Longest shortest route = %d, most_ports = %d",
	A.map_info->mi_longest_route,
	A.map_info->mi_most_ports);
  }

  /*
   * The following is a heuristic to see what max route len we should allow.
   * It is possible that on large x16 networks, there may exist len 5 routes
   * to everyone, but it would be bad to use them exclusively because that 
   * would create great hotspots.  Conversely, on a fabric made with x32s, we
   * do not necessarily want to allow len 7 routes, because that could end up
   * dropping down to an empty enclosure which is seen as a false spine.
   */
  /* On big x16 networks, max route len is 7, otherwise 5 */
  if (A.map_info->mi_most_ports <= 16) {
    if (fp->num_xbars > 128) {
      max_route_len[2] = 7;
    } else {
      max_route_len[2] = 5;
    }
  } else {
    max_route_len[2] = 5;
  }

  /* make sure we can at least get to everyone */
  if (max_route_len[2] < A.map_info->mi_longest_route) {
    max_route_len[2] = A.map_info->mi_longest_route;
  }

  /* record start time */
  route_start_time = time(NULL);

    /*
     * We're going to route in 3 passes to avoid crosstalk between 
     * route length classes (e.g. length 3 routes getting locked into
     * ruts created by len 7 routes).
     */
  fma_dfs_clear_nic_route_count(fp);

  for (i=0; i<LF_NUM_ELEM(max_route_len); ++i) {

    /* Clear all link loads */
    fma_dfs_clear_load(fp);

    for (route_num=0; route_num<num_routes; ++route_num) {
      for (n=0; n<A.myri->num_nics; ++n) {
	nip = A.myri->nic_info[n];
	for (port=0; port<nip->myri_info.num_ports; ++port) {
	  fma_dfs_start_port_route(nip, port, route_num, max_route_len[i]);
	}
      }
    }
  }

  fma_log("Routing took %d seconds", (int)(time(NULL) - route_start_time));
}
